//
//  GSPenProtocol.h
//  GlyphsCore
//
//  Created by Georg Seifert on 16.11.11.
//  Copyright 2011 schriftgestaltung.de. All rights reserved.
//

#import <Cocoa/Cocoa.h>
#import <GlyphsCore/GSNode.h>

@class GSLayer;
@class GSComponent;
@class FTPointArray;
@class GSShadow;
@class GSGradient;

@protocol GSPenProtocol

/** Begin a new sub path, set the current point to 'pt'. You must
end each sub path with a call to pen.closePath() or pen.endPath().
*/
- (void)moveTo:(NSPoint)pt;

/** Draw a straight line from the current point to 'pt'.*/
- (void)lineTo:(NSPoint)pt;

/** Draw a cubic bezier with an arbitrary number of control points.

The last point specified is on-curve, all others are off-curve
(control) points. If the number of control points is > 2, the
segment is split into multiple bezier segments.
*/
- (void)curveTo:(NSPoint)pt off1:(NSPoint)off1 off2:(NSPoint)off2;

/** Draw a whole string of quadratic curve segments.

The last point specified is on-curve, all others are off-curve
points.
*/
- (void)qCurveTo:(FTPointArray *)points;

/** Close the current sub path. You must call either pen.closePath()
or pen.endPath() after each sub path.
 */
- (void)closePath;

/** End the current sub path, but don't close it. You must call
either pen.closePath() or pen.endPath() after each sub path.
 */
- (void)endPath;

- (void)addComponent:(GSComponent *)component transformation:(NSAffineTransform *)transformation;

@property (readonly, nonatomic) NSPoint lastPoint;

@optional

- (void)append:(NSObject<GSPenProtocol> *)pen;

@property (nonatomic, readonly) BOOL hasContent;

@property (nonatomic, strong) NSAffineTransform *transform;

/**
To initialize stuff
*/
- (void)startGlyph:(GSLayer *)layer;

- (void)endGlyph:(GSLayer *)layer;

- (void)startPath:(NSDictionary *)attributes;

- (void)clear;

/// starts an empty new path
- (void)beginPath;

- (void)moveTo:(NSPoint)pt connection:(GSNodeType)connection selected:(BOOL)selected;

- (void)lineTo:(NSPoint)pt connection:(GSNodeType)connection selected:(BOOL)selected;

- (void)curveTo:(NSPoint)pt off1:(NSPoint)off1 off2:(NSPoint)off2 connection:(GSNodeType)connection selected:(BOOL)selected;
- (void)drawInPen:(NSObject<GSPenProtocol> *)pen;

- (void)moveTo:(NSPoint)pt objects:(NSArray *)objects;
- (void)lineTo:(NSPoint)pt objects:(NSArray *)objects;
- (void)curveTo:(NSPoint)pt off1:(NSPoint)off1 off2:(NSPoint)off2 objects:(NSArray *)objects;

- (void)setHints:(NSArray *)hints;

- (BOOL)ignoreSingleNodes;
- (void)setIgnoreSingleNodes:(BOOL)ignoreSingleNodes;

- (void)qCurveWithPoints:(NSPoint *)points count:(NSUInteger)count;

- (void)setLastPoint:(NSPoint)point;

/// color drawing
@property (nonatomic, strong) NSColor *fillColor;
@property (nonatomic, strong) NSColor *strokeColor;
@property (nonatomic) CGFloat lineWidth;
@property (nonatomic) GSLineCapStyle lineCap;
@property (nonatomic) NSLineJoinStyle lineJoin;

- (BOOL)closed;

- (void)setShadow:(GSShadow *)shadow;
- (void)setGradient:(GSGradient *)gradient bounds:(NSRect)bounds;
- (void)setIdentifier:(NSString *)identifier;
@end
